package fr.cnam.rdv;

import java.util.*;
import java.io.*;
import fr.cnam.util.*;

// Classe de définition d'un agenda
//    L'agenda contient des rendez-vous, des rendez-vous, des tâches à faire
//    Ces éléments sont lus dans des fichiers plats (pas d'utilisation de base de données)
//
public class Agenda
{
    String nomFichier;

    // La collection contenant les éléments de l'agenda
    //   ElementAgenda est la classe abstraite dont les rendez-vous et taches héritent
    //
    private ArrayList<ElementAgenda> elements;

    // Constructeur
    //
    public Agenda()
    {
        nomFichier = "";

        // Initialisation des attributs
        //
        this.elements = new ArrayList<ElementAgenda>();
    }

    // Enregistre dans le fichier binaire le contenu de l'agenda
    //  Chaque élément de l'agenda est écrit un par un
    //
    public void enregistrer(String nomFichier) throws Exception
    {
        File f = new File("../"+nomFichier);
        FileOutputStream fos = new FileOutputStream(f);
        DataOutputStream dos = new DataOutputStream(fos);
        
        dos.writeInt(elements.size());
        for(ElementAgenda e : elements)
            e.ecrire(dos);
        dos.close();

        this.nomFichier = nomFichier;
    }

    // Lecture des éléments de l'agenda dans un fichier bianire
    //
    public void ouvrir(String nomFichier) throws Exception
    {
        ArrayList<ElementAgenda> tmp = new ArrayList<ElementAgenda>();

        File f = new File("../"+nomFichier);
        FileInputStream fis = new FileInputStream(f);
        DataInputStream dis = new DataInputStream(fis);
        
        int nb = dis.readInt();
        for(int i=0;i<nb;i++)
            {
                String type = dis.readUTF();
                
                if (type.equals("RDV"))
                    {
                        RendezVous rdv = new RendezVous("","","","");
                        rdv.lire(dis);
                        tmp.add(rdv);
                    }
                if (type.equals("TACHE"))
                    {
                        Tache tache = new Tache("","",0);
                        tache.lire(dis);
                        tmp.add(tache);
                    }
            }
        dis.close();

        this.nomFichier = nomFichier;
        this.elements = tmp;
    }

    // La méthode charger devient deux methodes d'import
    //
    public void importerRdvs(String nomFichierRdv) throws Exception
    {
        // Lecture du fichier contenant les rendez vous
        //
        int num=0;
        String[] lignes = Terminal.lireFichierTexte("../"+nomFichierRdv);

        // On traite chaque ligne contenant soit un rendez-vous classique soit
        //  un rendez-vous périodique
        //
        for(String l:lignes)
            {
                try{
                    num++;
                    if (! siPeriodiqueLigne(l))
                        {
                            RendezVous rdv = RendezVous.decoderLigne(l);
                            elements.add(rdv);
                        }
                    else
                        {
                            RendezVousPeriodique rdv = RendezVousPeriodique.decoderLigne(l);
                            elements.add(rdv);
                        }
                }catch(Exception ex)
                    {
                        throw new Exception("Erreur de lecture ligne "+num);
                    }
            }
    }

    public void importerTaches(String nomFichierTache) throws Exception
    {
        int num=0;

        // Lecture du fichier contenant les rendez vous
        //
        String[] lignes = Terminal.lireFichierTexte("../"+nomFichierTache);

        // On traite chaque ligne contenant une tache
        //
        for(String l:lignes)
            {
                try{
                    num++;
                    Tache tache = Tache.decoderLigne(l);
                    elements.add(tache);
                }catch(Exception ex)
                    {
                        throw new Exception("Erreur de lecture ligne "+num);
                    }
            }
    }

    // Méthode static qui détermine si une ligne du fichier contient un rendez-vous periodique ou non 
    //
    private static boolean siPeriodiqueLigne(String ligne) throws Exception
    {
        StringTokenizer strtok = new StringTokenizer(ligne," ");
        String str = strtok.nextToken();
        if ( str.equals("journalier") ||
             str.equals("hebdomadaire") ||
             str.equals("mensuel") )
            return true;
        else
            return false;
    }

    // Méthode qui retourne sous la forme d'un texte de caractère le contenu de 
    //  l'agenda
    public String toString()
    {
        // On trie les éléments de l'agenda
        Collections.sort(elements);

        // Construction par concaténation du contenu de chaque élément de l'agenda
        // Chaque élément est numéroté afin de pouvoir choisir l'élément
        //
        String s = "";
        for(int i=0;i<elements.size();i++)
            s=s+
                "Numero : "+i+"\n"+
                elements.get(i).toString()+"\n--------------------------------\n";
        return s;
    }

    // Méthode qui ajoute un nouvel élément dans l'agenda
    //
    public void ajouterElement(ElementAgenda e)
    {
        elements.add(e);
    }

    // Méthode qui retourne sous la forme d'un texte le contenu de 
    //  l'agenda dont les éléments sont pour le jour donné
    //
    public String agendaJour(Calendar jour)
    {
        String s="";
        boolean ok=false;
        for(int i=0;i<elements.size();i++)
            {
                ElementAgenda e=elements.get(i);

                // On teste si l'élément est dans la période du jour
                // Cette méthode est implémentée dans chacune des classes
                //   RendezVous et Tache
                //
                if (e.siDansLaPeriode(jour))
                    {
                        s=s+"Numero : "+i+"\n"+e.toString();
                        s=s+"\n--------------------------------\n";
                        ok=true;
                    }
            }
        if (!ok) s = "PAS D'ELEMENT A CETTE DATE";
        return s;
    }

    // Méthode static qui convertit une chaine en date Calendar
    // Si la chaine ne contient pas une date valide alors
    //   la valeur de retour est null
    //
    static public Calendar stringToCalendar(String date)
    {
        Calendar cal=null;
        try
            {
                String[] t = date.split("/");
                
                cal = Calendar.getInstance();
                cal.clear();
                cal.set(Integer.parseInt(t[2]),
                        Integer.parseInt(t[1])-1,  // Les mois commence en 0
                        Integer.parseInt(t[0]));
                
                // Si la date saisie est correcte en format mais contient des valeurs entières
                //  qui ne correspondent pas valides
                if ( (cal.get(Calendar.YEAR)!=Integer.parseInt(t[2])) ||
                     (cal.get(Calendar.MONTH)!=Integer.parseInt(t[1])-1) ||
                     (cal.get(Calendar.DAY_OF_MONTH)!=Integer.parseInt(t[0])) )
                    {
                        cal=null;
                    }
            }
        catch(Exception ex){ cal = null;}  // Exception si erreur de format (pb de / ou pas un entier)
        
        return(cal);
    }

    // Retourne le i-ième élément de l'agenda 
    //
    public ElementAgenda getElement(int i)
    {
        return elements.get(i);
    }
}
