package fr.cnam.rdv;

import java.util.*;
import java.io.*;

import fr.cnam.ihm.*;

// Classe de définition d'un rendez-vous classique (non périodique)
// Elle hérite de la classe abstraite ElementAgenda
// Elle implémente l'interface FormulaireInt pour pouvoir modifier 
//  un RendezVous avec une IHM
//
public class RendezVous extends ElementAgenda implements FormulaireInt
{
    protected String date;       // Date de debut du rendez-vous
    protected String heureDebut; // Heure de debut du rendez-vous
    protected String heureFin;   // Heure de fin du rendez-vous

    // Constructeur
    //
    public RendezVous(String date,
                      String heureDebut,
                      String heureFin,
                      String texte)
    {
        super(texte);
        this.date       = date;
        this.heureDebut = heureDebut;
        this.heureFin   = heureFin;
    }

    // Méthode static qui retourne une RendezVous à partir d'une ligne du fichier texte
    //  Si la ligne ne peut pas etre décodée alors retourne un exception
    //
    public static RendezVous decoderLigne(String ligne) throws Exception
    {
        StringTokenizer strtok = new StringTokenizer(ligne," ");
        String date = strtok.nextToken();
        String heureDebut = strtok.nextToken();
        String heureFin = strtok.nextToken();
        String texte="";
        while (strtok.hasMoreTokens())
            texte=texte+" "+strtok.nextToken();
        
        return( new RendezVous(date,
                               heureDebut,
                               heureFin,
                               texte) );
    }

    // RendezVous sous forme de chaine
    //
    public String toString()
    {
        String str="";
        str = str + String.format("%-15s : %-15s\n","Date",date);
        str = str + String.format("%-15s : %-15s\n","Heure debut",heureDebut);
        str = str + String.format("%-15s : %-15s\n","Heure fin",heureFin);
        str = str + String.format("%-15s : %-15s\n","Texte",texte);
        return(str);
    }
    
    // Retourne la date du RendezVous sous la forme d'un Calendar
    //
    public Calendar getDate()
    {
        Calendar cal = Calendar.getInstance();
        cal.clear();
        try{
            String[] t = date.split("/");
            cal.set(Integer.parseInt(t[2]),
                    Integer.parseInt(t[1])-1,  // Les mois commence en 0
                    Integer.parseInt(t[0]));
        }catch(Exception ex){} // si erreur de format la date est à 0
        return(cal);
    }

    // Cette méthode détermine si le RendezVous est dans le jour donné
    //   (Cette même méthode est implémentée par RendzeVousPeriodique et Tache)
    //
    public boolean siDansLaPeriode(Calendar jour)
    {
        return getDate().equals(jour) ;
    }

    // Cette méthode est utilisée par la méthode compareTo qui est dans la classe abstraite
    //  ElementAgenda.
    // Elle retourne un entier qui est utilisé comme valeur de comparaison pour trier
    //  les éléments de l'agenda
    //
    public long getOrdre()
    {
        // La valeur entière d'une date (nombre de millisecondes depuis le 1/1/1970
        return( getDate().getTime().getTime() );
    }

    // Cette méthode est appelée dans l'IHM pour modifier un RendezVous
    // Elle est également implémentée dans chacune des classes RendezVousPeriodique et Tache
    //
    public void modifierIhm()
    {
        // Création du formulaire : l'objet RendezVous lui-même est passé en paramètre
        //   C'est lui qui réalise l'action
        Formulaire form = new Formulaire("Modifier un rendez vous",this,false,400,300);

        form.addText("DATE_RDV","Date",true,this.date);
        form.addText("HEURE_DEBUT_RDV","Heure de debut",true,this.heureDebut);
        form.addText("HEURE_FIN_RDV","Heure de fin",true,this.heureFin);
        form.addText("TEXTE","Texte",true,this.texte,-1,-1,-1,-1,250);
        form.addButton("MODIFIER","Modifier");

        try{
            form.afficher();
        }catch(Exception ex){}
    }

    // Méthode appelée par le formulaire quand on clique sur un bouton
    //
    public void  submit(Formulaire form,String nomSubmit)
    {
        if (nomSubmit.equals("MODIFIER"))
            {
                // On récupère les valeurs de l'IHM et on affecte les attributs de l'objet
                this.date       = form.getValeurChamp("DATE_RDV");
                this.heureDebut = form.getValeurChamp("HEURE_DEBUT_RDV");
                this.heureFin   = form.getValeurChamp("HEURE_FIN_RDV");
                this.texte      = form.getValeurChamp("TEXTE");
                form.fermer();
            }
    }
    

    public void ecrire(DataOutputStream dos) throws Exception
    {
        dos.writeUTF("RDV");
        dos.writeUTF(this.date);
        dos.writeUTF(this.heureDebut);
        dos.writeUTF(this.heureFin);
        dos.writeUTF(this.texte);
    }

    public void lire(DataInputStream dis) throws Exception
    {
        this.date       = dis.readUTF();
        this.heureDebut = dis.readUTF();
        this.heureFin   = dis.readUTF();
        this.texte      = dis.readUTF();
    }
}
