package fr.cnam.rdv;

import java.util.*;
import fr.cnam.ihm.*;

// Classe de définition d'un rendez-vous périodique
// Cette classe hérite de la classe RendezVous car elle la complète
// Cette classe implémente l'interface FormulaireInt qui est
//  utilisée par le formulaire de modification en IHM
//
public class RendezVousPeriodique extends RendezVous implements FormulaireInt
{
    private String periode;  // La période du rendez-vous : 
    // "journalier" "hebdomadaire" ou "mensuel"

    private String dateFin;  // Date de fin du rendez-vous périodique

    // Constructeur
    //
    public RendezVousPeriodique(String periode,
                                String dateDebut,
                                String dateFin,
                                String heureDebut,
                                String heureFin,
                                String texte)
    {
        super(dateDebut,heureDebut,heureFin,texte);
        this.periode = periode;
        this.dateFin = dateFin;
    }

    // Méthode static qui retourne une RendezVousPeriodique à partir d'une ligne du fichier texte
    //  Si la ligne ne peut pas etre décodée alors retourne un exception
    //
    public static RendezVousPeriodique decoderLigne(String ligne) throws Exception
    {
        StringTokenizer strtok = new StringTokenizer(ligne," ");
        String periode = strtok.nextToken();
        String dateDebut = strtok.nextToken();
        String dateFin = strtok.nextToken();
        String heureDebut = strtok.nextToken();
        String heureFin = strtok.nextToken();
        String texte="";
        while (strtok.hasMoreTokens())
            texte=texte+" "+strtok.nextToken();
        
        return( new RendezVousPeriodique(periode,
                                         dateDebut,
                                         dateFin,
                                         heureDebut,
                                         heureFin,
                                         texte) );
    }

    // RendezVousPeriodique en chaine
    //
    public String toString()
    {
        String str="";
        str = str + String.format("%-15s : %-15s\n","Periode",periode);
        str = str + String.format("%-15s : %-15s\n","Date de fin",dateFin);
        str = str + super.toString(); // Appel de la classe h&eacute;rit&eacute;e
        return(str);
    }

    // Retourne la date calendaire de début du rendez-vous
    //
    public Calendar getDateDebut()
    {
        return(getDate());
    }

    // Retourne la date calendaire de fin du rendez-vous
    //
    public Calendar getDateFin()
    {
        Calendar cal = Calendar.getInstance();
        cal.clear();
        String[] t = dateFin.split("/");
        cal.set(Integer.parseInt(t[2]),
                Integer.parseInt(t[1])-1,  // Les mois commence en 0
                Integer.parseInt(t[0]));
        return(cal);
    }

    // Cette méthode détermine si le RendezVousPeriodique est dans le jour donné
    // Elle calcule toutes les dates possibles du rdv périodique puis recherche 
    //   si la journée est parmi une de ces dates.
    // (Cette même méthode est aussi implémentée par RendzeVous et Tache)
    //
    public boolean siDansLaPeriode(Calendar jour)
    {
        // Collection de toutes les dates du rdv p&eacute;riodique
        ArrayList<Calendar> dates = new ArrayList<Calendar>();
        
        // Calcul dans le cas journalier sauf samedi et dimanche
        if (periode.equals("journalier"))
            {
                Calendar calCour = (Calendar)getDateDebut().clone();
                while (! calCour.after(getDateFin()))
                    {
                        int j = calCour.get(Calendar.DAY_OF_WEEK);
                        if ( (j!=Calendar.SATURDAY) && (j!=Calendar.SUNDAY) )
                            {
                                dates.add((Calendar)calCour.clone());
                            }

                        // jour de la date + 1
                        calCour.set(Calendar.DAY_OF_MONTH,
                                    calCour.get(Calendar.DAY_OF_MONTH)+1);
                    }
            }
            
        // Calcul dans le cas hebdomadaire
        if (periode.equals("hebdomadaire"))
            {
                Calendar calCour = (Calendar)getDateDebut().clone();
                while (! calCour.after(getDateFin()))
                    {
                        dates.add((Calendar)calCour.clone());

                        // jour de la date + 7
                        calCour.set(Calendar.DAY_OF_MONTH,
                                    calCour.get(Calendar.DAY_OF_MONTH)+7);
                    }
            }
            
        // Calcul pour le cas mensuel
        if (periode.equals("mensuel"))
            {
                Calendar calCour = (Calendar)getDateDebut().clone();
                while (! calCour.after(getDateFin()))
                    {
                        dates.add((Calendar)calCour.clone());

                        // mois de la date + 1
                        calCour.set(Calendar.MONTH,
                                    calCour.get(Calendar.MONTH)+1);
                    }
            }
            
        // On recherche si jour n'est pas parmi les dates du rdv p&eacute;riodique
        boolean dedans = false;
        for(Calendar c : dates)
            if (c.equals(jour))
                dedans=true;

        return(dedans);
    }
    
    // Cette méthode est utilisée par la méthode compareTo qui est dans la classe abstraite
    //  ElementAgenda.
    // Elle retourne un entier qui est utilisé comme valeur de comparaison pour trier
    //  les éléments de l'agenda
    //
    public long getOrdre()
    {
        // La valeur entière d'une date (nombre de millisecondes depuis le 1/1/1970
        return( getDateDebut().getTime().getTime() );
    }

    // Cette méthode est appelée dans l'IHM pour modifier un RendezVous
    // Elle est également implémentée dans chacune des classes RendezVous et Tache
    //
    public void modifierIhm()
    {
        Formulaire form = new Formulaire("Modifier un rendez vous",this,false,400,300);
        form.addText("PERIODE","Periode",true,this.periode);
        form.addText("DATE_DEBUT_RDV","Date de debut",true,this.date);
        form.addText("DATE_FIN_RDV","Date de fin",true,this.dateFin);
        form.addText("HEURE_DEBUT_RDV","Heure de debut",true,this.heureDebut);
        form.addText("HEURE_FIN_RDV","Heure de fin",true,this.heureFin);
        form.addText("TEXTE","Texte",true,this.texte,-1,-1,-1,-1,250);
        form.addButton("MODIFIER","Modifier");

        try{
            form.afficher();
        }catch(Exception ex){}
    }

    // Méthode appelée par le formulaire quand on clique sur un bouton
    //
    public void  submit(Formulaire form,String nomSubmit)
    {
        if (nomSubmit.equals("MODIFIER"))
            {
                this.periode    = form.getValeurChamp("PERIODE");
                this.date       = form.getValeurChamp("DATE_DEBUT_RDV");
                this.dateFin    = form.getValeurChamp("DATE_FIN_RDV");
                this.heureDebut = form.getValeurChamp("HEURE_DEBUT_RDV");
                this.heureFin   = form.getValeurChamp("HEURE_FIN_RDV");
                this.texte      = form.getValeurChamp("TEXTE");
                form.fermer();
            }
    }

}
