package fr.cnam.rdv;

import java.util.*;
import java.io.*;

import fr.cnam.ihm.*;

// Classe de définition d'une tache (qq chose à faire)
// Elle hérite de la classe abstraite ElementAgenda
// Elle implémente l'interface FormulaireInt pour pouvoir modifier 
//  une Tache avec une IHM
//
public class Tache extends ElementAgenda implements FormulaireInt
{
    private String dateButee;  // Date de butée de la tache
    // Cette date Peut être null
    private int priorite;      // Priorité de la tache
    // utilisé quand il n'existe pas de date de butée
    
    // Constructeur
    //
    public Tache(String texte,String dateButee, int priorite)
    {
        super(texte);
        this.dateButee = dateButee;
        this.priorite = priorite;
    }

    // Retourne la date butée en Calendar de la tache
    //
    public Calendar getDateButee()
    {
        Calendar cal = Calendar.getInstance();
        cal.clear();
        try{
            String[] t = dateButee.split("/");
            cal.set(Integer.parseInt(t[2]),
                    Integer.parseInt(t[1])-1,  // Les mois commence en 0
                    Integer.parseInt(t[0]));
        }catch(Exception ex){} // si erreur de format la date est à 0
        return(cal);
    }

    // Méthode static qui retourne une Tache à partir d'une ligne du fichier texte
    //  Si la ligne ne peut pas etre décodée alors retourne un exception
    //
    public static Tache decoderLigne(String ligne) throws Exception
    {
        StringTokenizer strtok = new StringTokenizer(ligne," ");
        String premier = strtok.nextToken();  // Premier mot de la ligne

        // Le premier mot de la ligne est soit un entier soit une date
        boolean siEntier=true;

        int priorite=0;    // Par defaut
        String dateButee;
        try{
            priorite = Integer.parseInt(premier); 
            // C'est un entier
        }catch(Exception ex){siEntier=false;}  // Ce n'est pas un entier
        
        if (siEntier)
            dateButee=null;
        else
            dateButee = premier;

        // Decodage du reste de la ligne qui est le texte
        String texte="";
        while (strtok.hasMoreTokens())
            texte=texte+" "+strtok.nextToken();
        
        // Création de la Tache
        return( new Tache(texte,
                          dateButee,
                          priorite) );
    }
    
    // Tache en chaine
    //
    public String toString()
    {
        String str="";
        if (dateButee!=null)
            str = str + String.format("%-15s : %-30s\n","Date butee",dateButee);
        else
            str = str + String.format("%-15s : %-30s\n","Priorite",priorite);
        str = str + String.format("%-15s : %-30s\n","Texte",texte);

        return(str);
    }
    

    // Cette méthode détermine si la Tache est dans le jour donné
    // Si une tache n'a pas de date de butée alors elle est toujours pour le jour donné
    // Sinon elle est dans le jour donné si la tache doit être faite dans les 5 jours
    //
    public boolean siDansLaPeriode(Calendar jour)
    {
        if (dateButee==null) return true;
        else
            {
                Calendar calDateButee = Calendar.getInstance();
                calDateButee.clear();
                calDateButee.set(jour.get(Calendar.YEAR),
                                 jour.get(Calendar.DAY_OF_MONTH)-5,
                                 jour.get(Calendar.MONTH));

                if (jour.after(calDateButee))
                    return true;
                else
                    return false;
            }
    }

    // Cette méthode est utilisée par la méthode compareTo qui est dans la classe abstraite
    //  ElementAgenda.
    // Elle retourne un entier qui est utilisé comme valeur de comparaison pour trier
    //  les éléments de l'agenda
    //
    // Pour une tache cette valeur est soit la valeur de priorité si il n'y a pas de
    //  date de butée sinon la valeur entière de la date de butée
    //
    public long getOrdre()
    {
        if (dateButee==null)
            return(priorite);
        else
            return( getDateButee().getTime().getTime() );
    }

    // Cette méthode est appelée dans l'IHM pour modifier une Tache
    // Elle est également implémentée dans chacune des classes RendezVousPeriodique et Tache
    //
    public void modifierIhm()
    {
        Formulaire form = new Formulaire("Modifier une tache",this,false,400,300);
        form.addText("DATE_BUTEE","Date butee",true,this.dateButee);
        form.addText("PRIORITE","Priorite",true,this.priorite+"");
        form.addText("TEXTE","Texte",true,this.texte,-1,-1,-1,-1,250);
        form.addButton("MODIFIER","Modifier");

        try{
            form.afficher();
        }catch(Exception ex){}
    }

    // Méthode appelée par le formulaire quand on clique sur un bouton
    //
    // Si la date de butée n'est pas saisi (vide) alors elle est non renseignée
    // Si la priorité n'est pas saisie alors la valeur est par défaut 0
    //
    public void  submit(Formulaire form,String nomSubmit)
    {
        if (nomSubmit.equals("MODIFIER"))
            {
                this.dateButee = form.getValeurChamp("DATE_BUTEE");
                if (this.dateButee.equals(""))
                    this.dateButee=null;
                String sprio = form.getValeurChamp("PRIORITE");
                if (sprio.equals(""))
                    this.priorite=0;
                else
                    this.priorite = Integer.parseInt(sprio);
                this.texte = form.getValeurChamp("TEXTE");
                form.fermer();
            }
    }
    
    public void ecrire(DataOutputStream dos) throws Exception
    {
        dos.writeUTF("TACHE");
        if (dateButee==null)
            dos.writeUTF("");
        else
            dos.writeUTF(this.dateButee);
        dos.writeInt(this.priorite);
        dos.writeUTF(this.texte);
    }

    public void lire(DataInputStream dis) throws Exception
    {
        this.dateButee = dis.readUTF();
        if (this.dateButee.equals(""))this.dateButee=null;
        this.priorite  = dis.readInt();
        this.texte     = dis.readUTF();
    }


}