// Exemple du DP Singleton

import java.util.*;

// Programme principal du test du factory
//
public class Exemple03
{
  public static void main(String[] args)
  {
    // Creation du factory via le SINGLETON
    //
    CarteGriseFactory usine = CarteGriseFactory.getInstance();
        
    // Exemple de creation d'un objet du factory
    //
    CarteGrise cg1 = usine.creerProduit("LAFONT",
                                        "PIERRE",
                                        "AM 613 HC");
    System.out.println(cg1.toString());

    // Deux autres crations :
    //
    usine.creerProduit("DUPONT",
                       "PAUL",
                       "SM 312 HC");

    usine.creerProduit("GARRIC",
                       "JEAN",
                       "AM 234 AZ");

    // Itration sur les produits du Factory
    //
    System.out.println("Itration sur les produits du Factory");
    for(CarteGrise cg:usine)
      {
        System.out.println(cg.toString());
        System.out.println("-----------------------");
      }
        
    // Pour dmontrer que c'est bien un singleton
    //
    CarteGriseFactory usine2 = CarteGriseFactory.getInstance();
    
    // Recherche dans le factory
    //
    while (true)
      {
        System.out.print("Recherche : ");
        String str = Terminal.lireString();
                
        ArrayList<CarteGrise> lcg = usine2.search(str);
        for(CarteGrise cg : lcg)
          {
            System.out.println("----------------");
            System.out.println(cg.toString());
          }

        //Changer l'adresse du premier lment recherch
        if (lcg.size() > 0)
          {
            System.out.print("Nouvelle adrrese  : ");
            String adresse = Terminal.lireString();
            lcg.get(0).setAdresse(adresse);
          }
      }
  }
}

//==========================================================
// Interface de la carte grise
//
interface CarteGrise
{
  public String toString();
  public String getNom();
  public String getPrenom();
  public String getNumeroPlaque();
  public Calendar get1ereMiseEnCirculation();
  public String getReference();
  public String getAdresse();
  public void setAdresse(String adresse);
}

//==========================================================
// Classe du Factory de carte grise
//
class CarteGriseFactory implements Iterable<CarteGrise>
{
  // Le singleton : l'usine unique de carte grise
  //
  static private CarteGriseFactory usine = null;

  // stockage des objets de l'usine 
  // Chaque carte-grise est stock dans une table de hashing dont la clef est 
  // une rfrence dtermine par la classe CarteGriseImpl.
  //
  private Hashtable<String,CarteGrise> cartegrises;
    
  // Constructeur de cration de l'usine
  //
  private CarteGriseFactory()
  {
    cartegrises = new Hashtable<String,CarteGrise>();
  }

  // Methode qui retourne le singleton
  //
  static public CarteGriseFactory getInstance()
  {
    if (usine==null) usine = new CarteGriseFactory();
    return usine;
  }
    
  // Creation d'une carte grise en fonction de certaines donnes.
  // La date de 1re mise en circulation est cre automatiquement.
  // Ainsi que le numro de rfrence interne.
  //
  public CarteGrise creerProduit(String nom,
                                 String prenom,
                                 String numeroPlaque)
  {
    CarteGrise cg;
        
    cg = new CarteGriseImpl(nom,prenom,numeroPlaque);
        
    cartegrises.put(cg.getReference(),cg);
        
    return(cg);
  }

  // Retourne un itrateur sur le factory
  //
  public Iterator<CarteGrise> iterator(){
    return Collections.list(cartegrises.elements()).iterator();
  }
    
  // Recherche le morceau de chaine dans chacun des attributs
  //  de la carte grise
  //
  public ArrayList<CarteGrise> search(String critere)
  {
    ArrayList<CarteGrise> lcg = new ArrayList<CarteGrise>();

    for(CarteGrise cg : cartegrises.values())
      {
        String str = cg.toString();
        if (str.toLowerCase().indexOf(critere.toLowerCase())!=-1)
          lcg.add(cg);
      }
    return lcg;
  }
}

//==========================================================
// Classe de la carte grise
//
class CarteGriseImpl implements CarteGrise
{
  private static long compteur=0;   // utilis pour la rfrenc eunique d'une cg

  private String nom;
  private String prenom;
  private String adresse;
  private String numeroPlaque;
  private Calendar date1ereMiseEnCirculation;
  private String reference;
    
  // Constructeur d'une carte grise
  //
  public CarteGriseImpl(String nom,
                        String prenom,
                        String numeroPlaque)
  {
    this.nom = nom;
    this.prenom = prenom;
    this.numeroPlaque = numeroPlaque;
    this.date1ereMiseEnCirculation = Calendar.getInstance();  
    // Remarque au passage : Calendar, c'est  la fois un Singleton et un factory
    compteur++;
    this.reference = String.format("%4d/%06d",
                                   this.date1ereMiseEnCirculation.get(Calendar.YEAR),
                                   compteur);
  }
    
  // Carte grise sous la forme d'une chaine de caractere
  //
  public String toString()
  {
    return(this.nom+"\n"+
           this.prenom+"\n"+
           this.adresse+"\n"+
           this.numeroPlaque+"\n"+
           this.date1ereMiseEnCirculation.getTime()+"\n"+
           this.reference);
  }

  // L'implmentation des methodes de l'interface
  //
  public String getNom(){return this.nom;}
  public String getPrenom(){return this.prenom;}
  public String getNumeroPlaque(){return this.numeroPlaque;}
  public Calendar get1ereMiseEnCirculation(){return this.date1ereMiseEnCirculation;}
  public String getReference(){return this.reference;}
  public String getAdresse(){return this.adresse;}
  public void setAdresse(String adresse){this.adresse=adresse;}
    
}
