package fr.cnam.horloge.model;

import java.util.*;
import java.awt.event.*;

import fr.cnam.horloge.controleur.ObservableHorloge;

//Classe de dfinition du modle de donnes de l'horloge.
//  Le modele de donne gre une heure courante que l'on 
//   peut changer et le nom de l'horloge qui l'on peut changer
//   aussi.
//  L'horloge gre un delta de temps par rapport  celui du temps
//   de l'horloge CPU.
//
// Cette classe utilise un Observable passe en parametre afin de prvenir les vues qui
//   se sont abonnes  la modification des donnes.
//   
public class ModelHorlogeImpl implements ModelHorloge, Runnable
{
    private String        nom;        // Nom de l'horloge
    private long          delta;      // Delta du temps avec CPU en ms
    private ObservableHorloge observable; // L'observateur
    private Thread        thread;
    private boolean       arret;

    // Constructeur de l'horloge.
    // Par defaut l'horloge est  l'heure de la CPU
    // 
        public ModelHorlogeImpl(String nom,ObservableHorloge observable)
    {
        this.nom = nom;
        this.observable = observable;
        delta = 0;
        thread = new Thread(this);
        thread.start();
        arret = true;
    }

    // Modification de l'heure de l'horloge
    //   
    public void majHeure(HeureHorloge heureHorloge)
    {
        observable.setChanged();

        // Mise a jour de la donne : recalcul du delta avec CPU
        Calendar temps = Calendar.getInstance();
        temps.set(Calendar.HOUR_OF_DAY,heureHorloge.getHeure());
        temps.set(Calendar.MINUTE,heureHorloge.getMinute());
        temps.set(Calendar.SECOND,heureHorloge.getSeconde());
        delta = temps.getTimeInMillis() - Calendar.getInstance().getTimeInMillis();
        
        // Notification aux abonnes
        notifier();
    }

    // Retourne l'observable du modele
    public ObservableHorloge getObservable()
    {
        return observable;
    }

    // Retourne l'heure de l'horloge en fonction du delta avec CPU
    //
    public HeureHorloge getHeure()
    {
        // Temps de l'horloge en ms
        long t = Calendar.getInstance().getTimeInMillis() + delta;

        // Conversion de t en date calendaire
        Calendar c = Calendar.getInstance();
        c.setTimeInMillis(t);

        // On retourne une donne lmentaire de l'heure de l'horloge
        return new HeureHorloge(c.get(Calendar.HOUR_OF_DAY),
                                c.get(Calendar.MINUTE),
                                c.get(Calendar.SECOND));
    }

    // Retourne le nom de l'horloge
    //
    public String getNom()
    {
        return nom;
    }

    // Change l'arret de l'horloge
    public void setArret(boolean value)
    {
        arret = value;
    }

    // Renommer le nom de l'horloge
    //
    public void renommer(String nom)
    {
        this.nom = nom;
        observable.setChanged();


        // Notification aux abonnes
        notifier();
    }

    // Notifier aux abonnes
    //
    public void notifier()
    {
        ArrayList args = new ArrayList();
        args.add("NOM");
        args.add(getNom());
        System.out.println("Notification du NOM");
        observable.notifyObservers(args);
        //
        observable.setChanged();
        args = new ArrayList();
        args.add("HEURE");
        args.add(getHeure());
        System.out.println("Notification de l'HEURE");
        observable.notifyObservers(args);
    }

    // Notification de l'heure de l'horloge
    //
    public void run()
    {
        while(true)
            {
                try{Thread.sleep(1000);}catch(Exception ex){};
                if (! arret) notifier();
            }
    }
}
