package fr.cnam.horloge.controleur;

import java.rmi.*;
import java.rmi.server.*;
import java.rmi.registry.*;
import java.util.*;

import fr.cnam.horloge.model.*;
import fr.cnam.horloge.controleur.*;
import fr.cnam.horloge.vue.ObserverHorloge;
import fr.cnam.horloge.vue.AdaptObserverHorloge;
import fr.cnam.horloge.vue.ObserverInt;

// Classe d'implmentation du controleur de l'horloge
// Elle implmente l'interface ControleurHorloge.
//
public class ControleurHorlogeImpl implements ControleurHorloge
{
  private ModelHorloge modelHorloge; // Modele de donne de l'horloge
    
  static private boolean distant = false;
  static private String  host;    // host du serveur contenant le controleur/modele
  static private int     port;    // port de l'adaptateur RMI du serveur
  static private int     portRMI; // port RMI utilise par le client

  // Constructeur
  // Le controleur doit connaitre le modele de donne utilis
  //
  public ControleurHorlogeImpl(ModelHorloge model)
  {
    this.modelHorloge = model;
  }

  // ---------- Les mthodes de l'interface ControleurHorloge
  // ============================================================
  public void actionRenommerHorloge(String nomHorloge)
  {
    if (! nomHorloge.equals(""))
      modelHorloge.renommer(nomHorloge);
  }
    
  public void actionChangerHeureHorloge(HeureHorloge heure)
  {
    if (heure!=null)
      modelHorloge.majHeure(heure);
  }

  public void actionStartHorloge()
  {
    modelHorloge.setArret(false);
  }

  public void actionStopHorloge()
  {
    modelHorloge.setArret(true);
  }


  public void actionNotifier()
  {
    modelHorloge.notifier();
  }


  // Mthode permettant de grer le controleur de l'horloge comme un
  //  singleton.
  // Cela reste la faon la plus simple de faire connaitre le controleur
  //  au programme principal
  //
  // Pour simplifier nous faisons le choix que le modele de donne est
  //   cr par le controleur
  //
  static public ControleurHorloge getInstance()
  {
    if (distant)
      {
        try{
          String addr = "rmi://"+host+":"+port+"/CONTROLEUR_HORLOGE";
          ControleurHorlogeODInt  cd = (ControleurHorlogeODInt)(Naming.lookup(addr));
          return new AdaptControleur(cd);
        }catch(Exception ex){System.out.println(ex+"\nImpossible lookup sur controleur: "
                                                +host+" "+port);};
      }

    if (modelHorlogeSingleton==null)
      {
        modelHorlogeSingleton  = new ModelHorlogeImpl("TOULOUSE", new ObservableHorloge() );
        controleurHorlogeSingleton = new ControleurHorlogeImpl(modelHorlogeSingleton);
      }
    return controleurHorlogeSingleton;
  }

  // Les attributs de singleton
  private static ModelHorloge modelHorlogeSingleton = null;
  private static ControleurHorloge controleurHorlogeSingleton = null;

  // Pour utiliser le controleur en remote
  //
  public static void setDistant(String... args)
  {
    // Port de l'OD ObserverHorlogeOD utilis pour tre notifier de
    //   manire distante
    // Si on cree plusieurs client sur la meme machine, il faut que ce port soit
    //  different
    // 
    int portRMI = Integer.parseInt(args[0]);
    
    // Host et port du controleur distant
    //
    String hostControleur = args[1];
    int portControleur = Integer.parseInt(args[2]);

    // Configuration du controleur afin de l'utiliser
    //   de manire distante
    //
    distant=true;
    host=hostControleur;
    port=portControleur;
    portRMI=portRMI;
  }

  // Pour sabonner soit en local soit en distant
  //
  public static void sabonner(ObserverInt observerVue)
  {
    Observer obsserver = new ObserverHorloge(observerVue);

    if(!distant)
      {
        ControleurHorlogeImpl.getInstance().getModelHorloge().getObservable().addObserver(obsserver);
      }
    else
      {
        try{
          new AdaptObserverHorloge(portRMI,
                                   obsserver,
                                   host,port);
        }catch(Exception ex){ex.printStackTrace();}
      }
  }


  // retourne le modele qui est gr par le controleur
  public ModelHorloge getModelHorloge()
  {
    return modelHorloge;
  }
}
