package fr.cnam.othello.vue;

import java.util.*;
import java.awt.*;
import java.awt.event.*;
import fr.cnam.othello.model.*;
import fr.cnam.othello.controleur.*;
import fr.cnam.ihm.*;
import fr.cnam.ia.*;

// Classe de definition du robot
//
public class RobotOthello implements AlphaBetaInt
{
  private   ControleurOthello controleur; // Le controleur
  private   int               numJoueurRobot;  // 1 ou 2

  // Constructeur
  //
  public RobotOthello(int numJoueurArg)
  {
    // Initialisation par defaut des attributs de la vue
    numJoueurRobot  = numJoueurArg;
    controleur = ControleurOthelloImpl.getInstance();
    controleur.addObserveur(new ObserverRobot(this));
  }

  // Le robot joue 
  //
  public void jouerCoup(DataVueOthello data)
  {
    int numJoueurCourant = controleur.getNumJoueurCourant();
    if (numJoueurRobot!=numJoueurCourant) 
      {
        System.out.println("Pas au robot");
        return;
      }

    int joueurAdverse;
    if (numJoueurRobot==1) joueurAdverse=2;
    else joueurAdverse=1;
    AlphaBeta algo = new AlphaBeta(this,
                                   5,  // On peut augmenter la profondeur
                                   numJoueurRobot,
                                   joueurAdverse);
    Point p = (Point)algo.estimer(new EtatJeuGrille(data.getGrille()));
    if (p==null) 
      {
        System.out.println("LA PARTIE EST TERMINEE");
        int diff = differencePions(data.getGrille(),numJoueurRobot,joueurAdverse);
        if (diff>0) System.out.println("LE ROBOT "+numJoueurRobot+" A GAGNE AVEC UNE DIFFERENCE DE "+diff);
        else if (diff<0) System.out.println("LE ROBOT "+numJoueurRobot+" A PERDU AVEC UNE DIFFERENCE DE "+diff);
        else System.out.println("MATCH NUL");
      }
    else 
      {
        System.out.println("Coup joue par le robot: "+p.x+" "+p.y);
        controleur.actionJouerOthello(numJoueurRobot,p.x,p.y);
      }
  }

  // Difference du nombre de pions entre ceux
  //  de numJoueur et l'autre
  //
  static public int differencePions(int[][] grille,int numJoueur,int numJoueurAdverse)
  {
    int s1=0;
    int s2=0;
    for(int i=0;i<ModelOthello.NB_COLONNES;i++)
      for(int j=0;j<ModelOthello.NB_LIGNES;j++)
        {
          if(grille[i][j]==numJoueur) s1++;
          if(grille[i][j]==numJoueurAdverse) s2++;
        }
    return s1-s2;
  }


  // retourne les coups valides d'un joueur en fonction d'une grille
  //
  public Vector coupsValides(int joueur,EtatJeu etatJeu)
  {
    EtatJeuGrille etatJeuGrille=(EtatJeuGrille)etatJeu;
    int[][] grille = etatJeuGrille.grille;
    Vector vlist =new Vector();
    
    int nb=0;
    for(int i=0;i<ModelOthello.NB_COLONNES;i++)
      for(int j=0;j<ModelOthello.NB_LIGNES;j++)
        if (grille[i][j]==0)
          {
            Vector v = ModelOthello.calculerPrises(grille,joueur,i,j);
            if (v.size()!=0) 
              vlist.add((Object)new Point(i,j));
          }
    return vlist;
  }

  // retourne une copie de la grille avec la mise a jour de la grille en
  // entree en fonction d'un coup jou
  public EtatJeu majEtatJeu(int joueur,EtatJeu etatJeu,Object coup)
  {
    EtatJeuGrille etatJeuGrille=(EtatJeuGrille)etatJeu;
    int[][] grille = etatJeuGrille.grille;
    int[][] grilleNouvelle = new int[ModelOthello.NB_COLONNES][ModelOthello.NB_LIGNES];
    for(int i=0;i<ModelOthello.NB_COLONNES;i++)
      for(int j=0;j<ModelOthello.NB_LIGNES;j++)
        grilleNouvelle[i][j]=grille[i][j];

    Point p = (Point)coup;
    Vector pionsARetourner = ModelOthello.calculerPrises(grilleNouvelle,joueur,p.x,p.y);
    if (pionsARetourner.size()!=0)
      {
        for(int i=0;i<pionsARetourner.size();i++)
          {
            Point pp = (Point)(pionsARetourner.elementAt(i));
            grilleNouvelle[(int)pp.x][(int)pp.y] = numJoueurRobot;
          }
      }
    return (EtatJeu)(new EtatJeuGrille(grilleNouvelle));
  }
  
  // evalue la "force" d'un tat de la grille sachant que le coup jou est "coup"
  // retourne une valeur numrique qui traduit l'avantage du joueur
  // Cette valeur est maximal si le joueur courant est le joueur automatique
  // Cette valeur est minimal sinon
  //
  //  joueurOrdinateur : Numero du joueur automatique
  //  joueur           : numero du joueur courant dans l'alpha beta
  //  joueurAdverse    : numero du joueur adverse de joueur
  //  coup             : coup jou (si ncessaire dans le calcul d'valuation)
  //  etatJeu          : tat de la grille du jeu suite au coup jou
  //
  public ResultatAlphaBeta evaluer(int joueurOrdinateur,int joueur,int joueurAdverse,Object coup,EtatJeu etatJeu)
  {
    EtatJeuGrille etatJeuGrille=(EtatJeuGrille)etatJeu;
    int[][] grille = etatJeuGrille.grille;
    int sens;
    if (joueur==joueurOrdinateur)sens=1;
    else sens=-1;

    if (coup!=null)
      {
        // Un coup dans les coins est prioritaire.
        // Cela signifie que l'algo AlphaBeta ne descend pas plus bas dans
        // la recherche d'un meilleur coup.
        //
        Point p = (Point)coup;
        if ( ((p.x==0) && (p.y==0))||
             ((p.x==0) && (p.y==ModelOthello.NB_LIGNES-1))||
             ((p.x==ModelOthello.NB_COLONNES-1) && (p.y==0))||
             ((p.x==ModelOthello.NB_COLONNES-1) && (p.y==ModelOthello.NB_LIGNES-1)) )
          return new ResultatAlphaBeta(100*sens,coup,1);  // 1 => prioritaire
        
        // Si le coup est sur un bord alors meilleur que la diffrence
        //  Poids important.
        //
        if ( (p.x==0) || (p.y==0)||
             (p.y==ModelOthello.NB_LIGNES-1)||
             (p.x==ModelOthello.NB_COLONNES-1) )
          return  new ResultatAlphaBeta(50*sens,coup,0);
      }

    // Diffrence des pions retourns par le joueur automatique avec l'adverse
    //
    if (joueur==joueurOrdinateur) return new ResultatAlphaBeta(differencePions(grille,joueur,joueurAdverse),coup,0);
    else return new ResultatAlphaBeta(differencePions(grille,joueurAdverse,joueur),coup,0);

  }
}

class EtatJeuGrille implements EtatJeu
{
  public int[][] grille;
  public EtatJeuGrille(int[][] grille)
  {
    this.grille=grille;
  }
}
